<?php

namespace App\Http\Controllers;
use App\Models\TypeEnseignement;
use App\Models\Serie;
use App\Models\SchoolSerie;
use App\Models\Setting;
use App\Models\SchoolTypeEnseignement;
use App\Models\School;
use App\Models\SchoolMatter;
use App\Models\FiliereSchool;
use App\Models\GetDren;
use Illuminate\Support\Facades\Storage;
use Illuminate\Http\Request;

class SettingController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        try{
            $school = Setting::where('school_id',auth()->user()->school_id)->first();
            return view('pages.setting.index', ['school' => $school]);
        }
        catch (\Exception $e) {
            return back()->with([
                'str' => 'danger',
                'msg' => 'Une erreur est survenue !'
            ]);
        }
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        try{
            $types = TypeEnseignement::where('status', '1')->get();
            $series = Serie::where('type', 'secondaire')->where('serie', '<>', 'A1')->where('serie', '<>', 'A2')->get();
            $dren = GetDren::where('status', '1')->orderby('libelle','ASC')->get();
            return view('pages.setting.create',[
                'school'  => null,
                'types' => $types,
                'series' => $series,
                'schoolType' => null,
                'schoolSerie' => null,
                'dren' => $dren
            ]);
        }
        catch (\Exception $e) {
            return back()->with([
                'str' => 'danger',
                'msg' => 'Une erreur est survenue !'
            ]);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'dren' => 'required|string',
            'emailSchool' => 'required|string|unique:settings,address_email',
            'telephonSchool' => 'required|numeric|unique:settings,phon_school',
            'addrePostal' => 'nullable|string|unique:settings,addres_postal',
            'citySchool' => 'required|string',
            'iepSchool' => 'nullable|string',
            'transport' => 'nullable|integer',
            'boxCash' => 'nullable|integer',
            'cantine' => 'nullable|integer',
            'classe' => 'nullable|integer'
        ]);

        if(!$request['type']){
            returnredirect()->route('setting.index')->with([
                'str' => 'danger',
                'msg' => 'Veuillez cocher au moins un type d\'enseignement !'
            ]);
        }

        if($request['series'] && $request['series'][0] == 1){
            $request['series'] = array_merge($request['series'], ['2','3']);
        }

        /** @var UploadedFile $img */ 
        if($request->img){
            $img = $request->validate(['img' => 'image', 'mimes:jpg,png,jpeg']);
            $val = explode('@', $request['emailSchool']);
            $name = 'logo_'.$val[0].'_'.date('Y').'.png'; // Debut du mail + année en cours
            $file = $img['img']->storeAs('logo', $name, 'public');
        }

        Setting::create([
            'get_dren_id' => $validated['dren'],
            'address_email' => $validated['emailSchool'],
            'phon_school' => $validated['telephonSchool'],
            'addres_postal' => $validated['addrePostal'],
            'address_iep' => $validated['iepSchool'],
            'school_city' => strtolower($validated['citySchool']),
            'box' => $request['boxCash'] ?? '0',
            'transport' => $request['transport'] ?? '0',
            'canteen' => $request['cantine'] ?? '0',
            'letter_class' => $request['classe'] ?? '0',
            'musique_aplastique'  => $request['autres'] ?? '0',
            'school_id' => auth()->user()->school_id,
            'image' => $file ?? null,
            'create_school' => $request['created'] ?? null,
            'open_school' => $request['opening'] ?? null,
        ]);
        foreach($request['type'] as $type){
            SchoolTypeEnseignement::create(['school_id' => auth()->user()->school_id, 'type_enseignement_id' => $type]); 
        }

        if($request['series']){
            foreach($request['series'] as $serie){
                SchoolSerie::create(['school_id' => auth()->user()->school_id, 'serie_id' => $serie]); 
            }
        }
        return redirect()->route('setting.index')->with([
            'str' => 'success',
            'msg' => 'Parametrage effectué !'
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        try{
            $school = Setting::find($id);
            $types = TypeEnseignement::where('status', '1')->get();
            $series = Serie::where('type', 'secondaire')->where('serie', '<>', 'A1')->where('serie', '<>', 'A2')->get();
            $schoolType = SchoolTypeEnseignement::where('school_id',$id)->get();
            $schoolSerie = SchoolSerie::where('school_id',$id)->get();
            $dren = GetDren::where('status', '1')->orderby('libelle','ASC')->get();
            return view('pages.setting.create',[
                'school'  => $school,
                'types' => $types,
                'series' => $series,
                'schoolType' => $schoolType,
                'schoolSerie' => $schoolSerie,
                'dren' => $dren
            ]);
        }
        catch (\Exception $e) {
            return back()->with([
                'str' => 'danger',
                'msg' => 'Une erreur est survenue !'
            ]);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $validated = $request->validate([
            'dren' => 'required|string',
            'emailSchool' => 'required|string',
            'telephonSchool' => 'required|numeric',
            'addrePostal' => 'nullable|string',
            'iepSchool' => 'nullable|string',
            'citySchool' => 'required|string',
            'boxCash' => 'nullable|integer',
            'transport' => 'nullable|integer',
            'cantine' => 'nullable|integer',
            'classe' => 'nullable|integer',
            'autres' => 'nullable|integer'
        ]);

        if(!$request['type']){
            return back()->with([
                'str' => 'danger',
                'msg' => 'Veuillez cocher au moins un type d\'enseignement'
            ]);
        }

        if($request['series'] && $request['series'][0] == 1){
            $request['series'] = array_merge($request['series'], ['2','3']);
        }

        $info = Setting::find($id);

        /** @var UploadedFile $img */ 
        if($request->img){
            $img = $request->validate(['img' => 'image', 'mimes:jpg,png,jpeg']);
            if ($info->image && Storage::exists('app/public/logo'.$info->image)) {
                Storage::delete('app/public/logo'.$info->image);
            }
            $val = explode('@', $request['emailSchool']);
            $name = 'logo_'.$val[0].'_'.date('Y').'.png';
            $file = $img['img']->storeAs('logo', $name, 'public');
        }

        // Supprimer les element des table de relations shool et type enseignement, cycle, serie
        $this->deteleGetType($info['school_id']);
        $this->deteleGetSerie($info['school_id']);
        $this->deleteMatterExist($info['school_id']);
        $this->deleteFiliere($info['school_id']);

        // Update info school setting end relation
        $info->update([
            'get_dren_id' => $validated['dren'],
            'address_email' => $validated['emailSchool'],
            'phon_school' => $validated['telephonSchool'],
            'addres_postal' => $request['addrePostal'] ?? null,
            'address_iep' => $request['iepSchool'] ?? null,
            'school_city' => strtolower($validated['citySchool']),
            'box' => $request['boxCash'] ?? '0',
            'transport' => $request['transport'] ?? '0',
            'canteen' => $request['cantine'] ?? '0',
            'letter_class' => $request['classe'] ?? '0',
            'musique_aplastique'  => $request['autres'] ?? '0',
            'image' => $file ?? $info['image'],
            'create_school' => $request['created'] ?? null,
            'open_school' => $request['opening'] ?? null,
        ]);

        foreach($request['type'] as $type){
            SchoolTypeEnseignement::create(['school_id' => auth()->user()->school_id, 'type_enseignement_id' => $type]); 
        }

        if($request['series']){
            foreach($request['series'] as $serie){
                SchoolSerie::create(['school_id' => auth()->user()->school_id, 'serie_id' => $serie]); 
            }
        }

        // Redirection sur la liste des établissement
        return redirect()->route('setting.index')->with([
            'str' => 'info',
            'msg' => 'Mis à jour effectué avec success !'
        ]);
    }


    protected function deteleGetType($school){
        SchoolTypeEnseignement::where('school_id', $school)->delete();
    }

    protected function deteleGetSerie($school){
        SchoolSerie::where('school_id', $school)->delete();
    }


    protected function deleteMatterExist($school){
        SchoolMatter::where('school_id', $school)->delete();
    }


    protected function deleteFiliere($school){
        FiliereSchool::where('school_id', $school)->delete();
    }
}
