<?php

namespace App\Http\Controllers;

use App\Models\Level;
use App\Models\Classe;
use App\Models\Devoir;
use App\Models\SchoolYear;
use App\Models\CuttingSchoolYear;
use App\Models\SchoolTypeEnseignement;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Carbon\Carbon;

class ProgramDevoirController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        try{
            $type = SchoolTypeEnseignement::where('school_id', auth()->user()->school_id)->get();
            $cutting = CuttingSchoolYear::where('type', 'general')->where('school_year_id', $this->yaerActif())->get();
            return view('pages.devoirs.index', [
                'data' => $this->getCutting($cutting),
                'levels' => $this->getLevel(verifySeconde($type))
            ]);
        }
        catch(\Exception $e){
            return back()->with([
                'str' => 'danger',
                'msg' => 'Une erreur est survenue !'
            ]);
        }
    }


    public function getClass(Request $request)
    {
        try{
            $dts = Classe::where('level_id', $request['id'])->where('school_year_id', $this->yaerActif())->where('school_id', auth()->user()->school_id)->get();
            return response()->json([
                'status' => count($dts) ? 200:201,
                'data' => $dts ?? []
            ]);
        }
        catch(\Exception $e){
            return back()->with([
                'str' => 'danger',
                'msg' => 'Une erreur est survenue !'
            ]);
        }
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {
        try{
            $val = $request->validate([
                'cutting' => 'required|integer',
                'level' => 'required|integer',
                'classe' => 'required|integer',
            ]);
            $class = Classe::find($val['classe']);
            return View('pages.devoirs.create',[
                'matters' => $this->getMatiere($class),
                'classe' => $class,
                'cutting' => CuttingSchoolYear::find($val['cutting'])
            ]);
        }
        catch(\Exception $e){
            return back()->with([
                'str' => 'danger',
                'msg' => 'Une erreur est survenue !'
            ]);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try{
            $val = $request->validate([
                'cutting' => 'required|integer',
                'matters' => 'required|array',
                'matters.*' => 'required|integer',
                'dates' => 'required|array',
                'dates.*' => 'required|date',
                'debuts' => 'required|array',
                'debuts.*' => 'required|date_format:H:i',
                'fins' => 'required|array',
                'fins.*' => 'required|date_format:H:i',
                'choix' => 'required|string',
                'class' => 'required|string'
            ]);            
            $choix = explode(',', $val['choix']);
            if((count($val['matters']) == count($choix))){
                $i = 0;
                while($i < sizeof($choix)){
                    if(isset($val['matters'][$i]) && isset($val['dates'][$i]) && isset($val['debuts'][$i]) && isset($val['fins'][$i])){
                        $exist = $this->verifyData($val['matters'][$i], $val['class'], $val['cutting'], $val['dates'][$i], $choix[$i]);
                        if(!$exist){
                            Devoir::create([
                                'period' => $choix[$i],
                                'date' => $val['dates'][$i],
                                'debut' => $val['debuts'][$i],
                                'jour' => $this->getDay($val['dates'][$i]),
                                'fins' => $val['fins'][$i],
                                'matter_id' => $val['matters'][$i],
                                'classe_id' => $val['class'],
                                'school_id' => auth()->user()->school_id,
                                'school_year_id' => $this->yaerActif(),
                                'cutting_school_year_id' => $val['cutting'],
                            ]);
                        }
                    }
                    $i++;
                }
                return to_route('program.index')->with([
                    'str' => 'success',
                    'msg' => 'Programme de devoirs ajoutés.'
                ]);
            }
            else{
                return back()->with([
                    'str' => 'danger',
                    'msg' => 'Une erreur est survenue !'
                ]);
            }
        }
        catch(\Exception $e){
            return back()->with([
                'str' => 'danger',
                'msg' => 'Une erreur est survenue !'
            ]);
        }
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Request $request)
    {
        try{
            $dts = Devoir::find($request['id']);
            $table = [
                'classe' => $dts->classe->libelle,
                'matter' => $dts->matter->libelle,
                'code' => $dts->matter->abbreviated,
                'date' => date('d/m/Y', strtotime($dts->date)),
                'edit' => $dts->date,
                'debut' => $dts->debut,
                'fin' => $dts->fins,
                'period' => $dts->period
            ];
            return response()->json([
                'status' => $dts ? 200:201,
                'data' => $dts ? $table:[]
            ]);
        }
        catch(\Exception $e){
            return back()->with([
                'str' => 'danger',
                'msg' => 'Une erreur est survenue !'
            ]);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request)
    {
        try{
            $val = $request->validate([
                'id' => 'required|integer',
                'period' => 'required|integer',
                'date' => 'required|date',
                'debut' => 'required|date_format:H:i',
                'fin' => 'required|date_format:H:i'
            ]);

            $dts = Devoir::find($val['id']);
            if($dts){
                $dts->update([
                    'date' => $val['date'],
                    'jour' => $this->getDay($val['date']),
                    'debut' => $val['debut'],
                    'fins' => $val['fin'],
                    'period' => $val['period']
                ]);
            }
            return back()->with([
                'str' => 'warning',
                'msg' => 'Devoir de '.$dts->matter->abbreviated.' modifié en '.$dts->classe->libelle
            ]);
        }
        catch(\Exception $e){
            return back()->with([
                'str' => 'danger',
                'msg' => 'Une erreur est survenue !'
            ]);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Request $request)
    {
        try{
            $dts = Devoir::find($request['id']);
            if($dts){
                $dts->delete();
                return to_route('program.index')->with([
                    'str' => 'success',
                    'msg' => 'Devoir de '.$dts->matter->abbreviated.' supprimé en '.$dts->classe->libelle
                ]);
            }
        }
        catch(\Exception $e){
            return back()->with([
                'str' => 'danger',
                'msg' => 'Une erreur est survenue !'
            ]);
        }
    }



    protected function getCutting($dts)
    {
        $tables = []; $i = 1;
        $data = [1 => 'realtime-reorder', 2 => 'basic-col-reorder', 3 => 'saving-reorder'];
        foreach($dts as $item){
            $tables[] = [
                'id' => $item->id,
                'libelle' => $item->cutting['libelle'],
                'actif' => $item->status,
                'tableId' => $data[$i],
                'datas' => Devoir::where('cutting_school_year_id', $item->id)->where('school_id', auth()->user()->school_id)->orderBy('date')->get()
            ];
            $i++;
        }
        return $tables;
    }


    private function getLevel($valeurs){
        $lib = getIdLevel($valeurs);
        if(sizeof($lib) == 2){
            $dts = Level::where('type_enseignement_id', $lib[0])->orWhere('type_enseignement_id', $lib[1])->get();
        }
        else{
            $dts = $lib[0] == 6 ? auth()->user()->school->levels:Level::where('type_enseignement_id', $lib[0])->get();
        }
        return $dts;
    }


    private function getDay($day){
        $date = Carbon::parse($day);
        Carbon::setLocale('fr');
        return $date->translatedFormat('l');
    }

    private function yaerActif(){
        $year = SchoolYear::where('status', '1')->first();
        return $year->id;
    }


    private function verifyData($matter, $class, $cutting, $date, $period){
        $count = Devoir::where('classe_id',  $class)->where('matter_id', $matter)
        ->where('cutting_school_year_id', $cutting)->where('date', $date)->where('period', $period)->count();
        return $count;
    }


    public function getMatiere($class)
    {
        // $class = Classe::find($class);
        if($class['additif']){
            $additif = $class['additif'] == 'AP' ? 'Mus':'AP';
            if($class['lv2'] == 'mixte'){
                $dts = DB::table('matters')
                ->join('level_matter', 'matters.id', '=', 'level_matter.matter_id')
                ->select('matters.id', 'matters.libelle', 'matters.abbreviated')
                ->where('level_matter.school_id', '=', auth()->user()->school_id)
                ->where('level_matter.level_id', '=', $class['level_id'])
                ->where('level_matter.serie_id', '=', $class['serie_id'] ?? null)
                ->where('abbreviated', '<>', $additif)
                ->where('matters.libelle', '<>', 'mixte')
                ->where('level_matter.matter_id', '<>', '15')
                ->orderBy('matters.libelle')
                ->get();
            }
            else{
                $lv2 = $class['lv2'] == 'allemand' ? 'espagnol':'allemand';
                $dts = DB::table('matters')
                ->join('level_matter', 'matters.id', '=', 'level_matter.matter_id')
                ->select('matters.id', 'matters.libelle', 'matters.abbreviated')
                ->where('level_matter.school_id', '=', auth()->user()->school_id)
                ->where('level_matter.level_id', '=', $class['level_id'])
                ->where('level_matter.serie_id', '=', $class['serie_id'] ?? null)
                ->where('matters.libelle', '<>', $lv2 ?? null)
                ->where('abbreviated', '<>', $additif)
                ->where('matters.libelle', '<>', 'mixte')
                ->where('level_matter.matter_id', '<>', '15')
                ->orderBy('matters.libelle')
                ->get();
            }
        }
        else{
            if($class['lv2'] == 'mixte'){
                $dts = DB::table('matters')
                ->join('level_matter', 'matters.id', '=', 'level_matter.matter_id')
                ->select('matters.id', 'matters.libelle', 'matters.abbreviated')
                ->where('level_matter.school_id', '=', auth()->user()->school_id)
                ->where('level_matter.level_id', '=', $class['level_id'])
                ->where('level_matter.serie_id', '=', $class['serie_id'] ?? null)
                ->where('libelle', '<>','Arts plastique')->where('libelle', '<>','Musique')
                ->where('matters.libelle', '<>', 'mixte')
                ->where('level_matter.matter_id', '<>', '15')
                ->orderBy('matters.libelle')
                ->get();
            }
            else{
                $lv2 = $class['lv2'] == 'allemand' ? 'espagnol':'allemand';
                $dts = DB::table('matters')
                ->join('level_matter', 'matters.id', '=', 'level_matter.matter_id')
                ->select('matters.id', 'matters.libelle', 'matters.abbreviated')
                ->where('level_matter.school_id', '=', auth()->user()->school_id)
                ->where('level_matter.level_id', '=', $class['level_id'])
                ->where('level_matter.serie_id', '=', $class['serie_id'] ?? null)
                ->where('matters.libelle', '<>', $lv2 ?? null)
                ->where('libelle', '<>','Arts plastique')->where('libelle', '<>','Musique')
                ->where('matters.libelle', '<>', 'mixte')
                ->where('level_matter.matter_id', '<>', '15')
                ->orderBy('matters.libelle')
                ->get();
            }
        }
        return $dts;
    }
}
