<?php

namespace App\Http\Controllers;

use App\Models\Setting;
use App\Models\Evaluated;
use App\Models\EvaluatedLevel;
use App\Models\SchoolYear;
use App\Models\Classe;
use App\Models\LevelPrimaryMatters;
use App\Models\PrimaryNote;
use App\Models\ResultatPrimaire;
use Illuminate\Http\Request;
use App\Exports\CompositionStudentNoteExport;
use App\Imports\NotePrimaireImport;
use App\Jobs\CalulMoyennePrimaireJob;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\DB;

class CompositionController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try{
            $vals = $request->validate([
                'select' => 'required|string'
            ]);
            $val = explode('_', $vals['select']);
            $class = Classe::find($val[1]);
            $matters = LevelPrimaryMatters::where('level_id', $class['level_id'])->get();
            $evaluated = EvaluatedLevel::find($val[0]);
            return view('pages.composition.index',[
                'evaluated' => $evaluated,
                'classe' => $class,
                'matters' => $matters,
                'datas' => $this->getNoteEvaluated($val[1], $val[0], $matters)
            ]);
        }
        catch(\Exception $e){
            return back()->with([
                'str' => 'danger',
                'msg' => 'Une erreur est survenue !'
            ]);
        }
    }

    /**
     * Display a listing of the resource.
     */
    public function return(string $string)
    {
        try{
            $val = explode('_', $string);
            $class = Classe::find($val[1]);
            $matters = LevelPrimaryMatters::where('level_id', $class['level_id'])->get();
            $evaluated = EvaluatedLevel::find($val[0]);
            return view('pages.composition.index',[
                'evaluated' => $evaluated,
                'classe' => $class,
                'matters' => $matters,
                'datas' => $this->getNoteEvaluated($val[1], $val[0], $matters)
            ]);
        }
        catch(\Exception $e){
            return back()->with([
                'str' => 'danger',
                'msg' => 'Une erreur est survenue !'
            ]);
        }
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        try{
            $school = Setting::where('school_id',auth()->user()->school_id)->first();
            $prescolaire = $school->school->typeEnseignements->where('type_enseignement', 'prescolaire')->count();
            $primaire = $school->school->typeEnseignements->where('type_enseignement', 'primaire')->count();
            return response()->json(['prescolaire' => $prescolaire, 'primaire' => $primaire]);
        }
        catch(\Exception $e){
            return back()->with([
                'str' => 'danger',
                'msg' => 'Une erreur est survenue !'
            ]);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try{
            $request['typeEval'] =  $request['typeEval'] == 1 ? null:$request['typeEval']; 
            $vals = $request->validate([
                'typeEval' => 'required|string',
                'dateEval' => 'required|date'
            ]);
            $date = explode('-', $vals['dateEval']);
            $libelle = $vals['typeEval'].' '.$this->getMois($date[1]).' '.$date[0];
            $verify = Evaluated::where('libelle', $libelle)->where('date', $vals['dateEval'])->first() ?? $this->saveEvaluated($libelle, $vals['dateEval']);
            if($verify){
                EvaluatedLevel::create([
                    'evaluated_id' => $verify['id'],
                    'school_id' => auth()->user()->school_id,
                    'school_year_id' => $this->yearActif()['id'],
                    'ps' => $request['ps'] ?? '0',
                    'ms' => $request['ms'] ?? '0',
                    'gs' => $request['gs'] ?? '0',
                    'cp1' => $request['cp1'] ?? '0',
                    'cp2' => $request['cp2'] ?? '0',
                    'ce1' => $request['ce1'] ?? '0',
                    'ce2' => $request['ce2'] ?? '0',
                    'cm1' => $request['cm1'] ?? '0',
                    'cm2' => $request['cm2'] ?? '0',
                ]);
                return back()->with([
                    'str' => 'success',
                    'msg' => 'Enregistrement effectué.'
                ]);
            }
            else{
                return back()->with([
                    'str' => 'danger',
                    'msg' => 'Une erreur est survenue !'
                ]);
            }
        }
        catch(\Exception $e){
            return back()->with([
                'str' => 'danger',
                'msg' => 'Une erreur est survenue !'
            ]);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Request $request)
    {
        try{
            $tab = ['ps', 'ms', 'gs', 'cp1', 'cp2', 'ce1', 'ce2', 'cm1', 'cm2'];
            $data = EvaluatedLevel::where($tab[$request['level']-1], '1')->where('school_id', auth()->user()->school_id)->where('school_year_id', $this->yearActif()['id'])->orderBy('created_at', 'DESC')->get();

            return response()->json($this->getEvaluted($data));
        }
        catch(\Exception $e){
            return back()->with([
                'str' => 'danger',
                'msg' => 'Une erreur est survenue !'
            ]);
        }
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function export($string)
    {
        try{
            $val = explode('_', $string);
            $class = Classe::find($val[1]);
            $evaluated = EvaluatedLevel::find($val[0]);
            $name = $class->libelle.'_'.str_replace(' ', '_', ucwords($evaluated->evaluated->libelle));
            return Excel::download(new CompositionStudentNoteExport($val[1], $val[0]), $name.'.xlsx');
        }
        catch(\Exception $e){
            return to_route('composition.return',$string)->with([
                'str' => 'danger',
                'msg' => 'Une erreur est survenue !'
            ]);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function import(Request $request)
    {
        try{
            $val = $request->validate([
                'files' => 'required|mimes:xlsx',
                'class' => 'required|string',
                'evaluated' => 'required|string'
            ]);
            $evaluated = EvaluatedLevel::find($val['evaluated']);
            $class = Classe::find($val['class']);
            if($evaluated && $class){
                Excel::import(new NotePrimaireImport($val['class'], $val['evaluated']), $request->file('files'));
                dispatch(new CalulMoyennePrimaireJob($class['id'], $class['school_id'], $class['school_year_id'], $val['evaluated']));

                return to_route('composition.return',$request['evaluated'].'_'.$request['class'])->with([
                    'str' => 'info',
                    'msg' => 'Notes ajoutée pour cette classe !'
                ]);
            }
            else{
                return to_route('composition.return',$request['evaluated'].'_'.$request['class'])->with([
                    'str' => 'danger',
                    'msg' => 'Une erreur est survenue !'
                ]); 
            }
        }
        catch(\Exception $e){
            return to_route('composition.return',$request['evaluated'].'_'.$request['class'])->with([
                'str' => 'danger',
                'msg' => 'Une erreur est survenue !'
            ]);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }

    private function saveEvaluated($libelle, $date)
    {
        $data = Evaluated::create([
            'libelle' => $libelle,
            'date' => $date,
        ]);
        return $data;
    }


    private function getEvaluted($table)
    {
        $tab = [];
        foreach($table as $item){
            $tab[] = [
                'id' => $item->id,
                'libelle' => ucwords($item->evaluated->libelle),
                'abbreviated' => ucwords($item->evaluated->libelle)
            ];
        }

        return $tab;
    }

    private function getNoteEvaluated($class, $evalId, $matters)
    {
        $student = $this->getStudentClasse($class);
        $tabs = [];
        foreach($student as $item){
            $tabs[] = [
                'id' => $item->id,
                'name' => strtoupper($item->first_name).' '.ucwords($item->last_name),
                'sexe' => strtoupper($item->sexe),
                'matricule' => $item->matricule,
                'resultat' => ResultatPrimaire::where('inscription_id', $item->id)->where('evaluated_level_id', $evalId)->first(),
                'notes' => $this->noteStudent($item->id, $evalId, $matters)
            ];
        }
        return $tabs;
    }

    /**
     * Get notes student the specified resource from storage.
     */
    protected function getStudentClasse($id)
    {
        $year = $this->yearActif();
        $data = DB::table('students')
            ->join('inscriptions', 'students.id', '=', 'inscriptions.student_id')
            ->select('inscriptions.id', 'students.first_name', 'students.last_name', 'students.matricule', 'students.sexe', 'inscriptions.redoublant')
            ->where('inscriptions.classe_id', '=', $id)
            ->where('inscriptions.school_year_id', '=', $year->id)
            ->orderBy('students.first_name')->orderBy('students.last_name')
            ->get();
        return $data;
    }


    private function noteStudent($id, $evalId, $matters)
    {
        $data = [];
        foreach($matters as $matter){
            $data[] = [
                'note' => PrimaryNote::where('inscription_id', $id)->where('evaluated_level_id', $evalId)->where('level_primary_matter_id', $matter->id)->first(),
                'valeur' => $matter->points
            ];
        }
        return $data;
    }


    private function getMois($mois)
    {
        $datas = [
            '01' => "Janvier", '02' => "Février", '03' => "Mars", '04' => "Avril", 
            '05' => "Mai", '06' => "Juin", '07' => "Juillet", '08' =>"Août", 
            '09' => "Septembre", '10' => "Octobre", '11' => "Novembre", '12' => "Décembre" 
        ];
        return $datas[$mois];
    }

    /**
     * Get school year actif the specified resource from storage.
     */
    private function yearActif(){
        $dts = SchoolYear::where('status', '1')->first();
        return $dts;
    }
}
