<?php

use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\Request;
use App\Models\Student;
use App\Models\TimeMorning;
use App\Models\TimeAfter;
use App\Models\GetNote;
use App\Http\Controllers\NoteController;
use App\Http\Controllers\DevoirController;
use Illuminate\Support\Facades\DB;


/*
|--------------------------------------------------------------------------
| 🛡️ Fonction de sécurité : Vérifie si l'utilisateur est connecté
|--------------------------------------------------------------------------
| Cette fonction est utilisée pour protéger les routes de l'application.
| Si l'utilisateur n'est pas connecté, il est redirigé vers la page
| de connexion principale du portail EDUCAS-NSA.
|--------------------------------------------------------------------------
*/
function ensureAuthenticated()
{
    if (!Auth::check()) {
        redirect('https://educas-nsa.net/academie/connexion')->send();
        exit;
    }
}

/*
|--------------------------------------------------------------------------
| 🏠 Route d’accueil protégée
|--------------------------------------------------------------------------
| Affiche la page d’accueil personnalisée de l’élève connecté.
| - Vérifie l’authentification
| - Récupère les informations de l’élève et de son école
| - Transmet les données à la vue "welcome"
|--------------------------------------------------------------------------
*/
Route::get('/', function () {
    ensureAuthenticated();

    $student = Auth::user();
    $school  = $student->school()->with('setting')->first();

    return view('welcome', compact('student', 'school'));
})->name('welcome');

/*
|--------------------------------------------------------------------------
| 📘 Route du premier trimestre
|--------------------------------------------------------------------------
| Permet d’afficher la page du 1er trimestre.
| Les données affichées peuvent être personnalisées
| dans la vue "trimestre1.blade.php".
|--------------------------------------------------------------------------
*/
Route::get('/trimestre1', function () {
    ensureAuthenticated();

    $student = Auth::user();
    $school  = $student->school()->with('setting')->first();

    return view('trimestre1', compact('student', 'school'));
})->name('trimestre1');

/*
|--------------------------------------------------------------------------
| 📘 Route du deuxième trimestre
|--------------------------------------------------------------------------
| Permet d’afficher la page du 1er trimestre.
| Les données affichées peuvent être personnalisées
| dans la vue "trimestre1.blade.php".
|--------------------------------------------------------------------------
*/

Route::get('/trimestre2', function () {
    ensureAuthenticated();

    $student = Auth::user();
    $school  = $student->school()->with('setting')->first();

    return view('trimestre2', compact('student', 'school'));
})->name('trimestre2');

/*
|--------------------------------------------------------------------------
| 📘 Route du programme des devoirs
|--------------------------------------------------------------------------
| Permet d’afficher la page du programme des devoirs.
| Les données affichées peuvent être personnalisées
|--------------------------------------------------------------------------
*/




Route::get('/devoirs', [DevoirController::class, 'index'])->name('devoirs.index');
/*
|--------------------------------------------------------------------------
| 🧮 Route d’affichage générale des notes
|--------------------------------------------------------------------------
| Cette route :
| - Vérifie l’authentification
| - Récupère les notes de l’élève connecté
| - Charge les relations nécessaires (matière, trimestre, etc.)
| - Regroupe les notes par trimestre (cutting_id)
| - Envoie les données à la vue "notes.blade.php"
|--------------------------------------------------------------------------
*/
Route::get('/notes', function () {
    ensureAuthenticated();

    $student = Auth::user();
    $school  = $student->school()->with('setting')->first();
    $inscriptionId = $student->inscription_id ?? null;

    // Récupération des notes avec relations et groupement par trimestre
    $notes = GetNote::with([
            'evaluation.cutting',
            'evaluation.levelMatter.matter'
        ])
        ->where('inscription_id', $inscriptionId)
        ->get()
        ->groupBy(fn($note) => $note->evaluation->cutting->cutting_id);

    return view('notes', compact('student', 'school', 'notes'));
})->name('notes');

/*
|--------------------------------------------------------------------------
| 🧾 Route d’affichage des notes du 1er trimestre (via contrôleur)
|--------------------------------------------------------------------------
| Cette route utilise le contrôleur NoteController pour plus de clarté.
| Exemple de méthode :
|   NoteController::showTrimestre1($studentId)
|--------------------------------------------------------------------------
*/


Route::get('/notes/trimestre1', function () {
    $student = Auth::user();

    $school = $student->school()->with('setting')->first();
    $inscription = $student->inscriptions()->latest()->first();
    if (!$inscription) return "Aucune inscription trouvée pour cet élève.";

    $trimestre = DB::table('cuttings')->where('libelle', 'Trimestre 1')->first();
    if (!$trimestre) return "Trimestre 1 non défini.";

    $pdo = DB::connection()->getPdo();

    // Fonction pour détecter la base automatique
    $detectBase = function ($type) {
        $type = strtolower($type);

        if (in_array($type, ['interro', 'interrogation'])) return 10;
        if (in_array($type, ['composition', 'examen'])) return 40;
        if (in_array($type, ['devoir', 'controle', 'contrôle'])) return 20;

        return 20; // base par défaut
    };

    // -----------------------
    // Récupération des notes
    // -----------------------
    $sql = "
    SELECT 
        m.id AS matter_id,
        m.libelle AS matiere,
        e.sub_matter_id,
        CASE e.sub_matter_id
            WHEN 1 THEN 'Composition Française'
            WHEN 2 THEN 'Orthographe-Grammaire'
            WHEN 3 THEN 'Expression Orale'
            ELSE NULL
        END AS sous_matiere,
        et.type AS evaluation_type,
        gn.note,
        e.notee,
        e.date_create,
        coef.valeur AS coefficient,
        lm.id AS level_matter_id
    FROM get_notes gn
    INNER JOIN evaluations e ON gn.evaluation_id = e.id
    INNER JOIN level_matter lm ON e.level_matter_id = lm.id
    INNER JOIN matters m ON lm.matter_id = m.id
    INNER JOIN coefficients coef ON lm.coefficient_id = coef.id
    INNER JOIN evaluation_types et ON e.evaluation_type_id = et.id
    INNER JOIN cutting_school_year csy ON e.cutting_school_year_id = csy.id
    WHERE gn.inscription_id = :inscription_id
      AND csy.cutting_id = :trimestre_id
    ORDER BY m.libelle ASC, e.sub_matter_id ASC, e.id ASC
    ";

    $stmt = $pdo->prepare($sql);
    $stmt->execute([
        'inscription_id' => $inscription->id,
        'trimestre_id' => $trimestre->id
    ]);

    $rows = $stmt->fetchAll(PDO::FETCH_OBJ);

    if (empty($rows)) {
        return view('notes.trimestre1', [
            'student' => $student,
            'school' => $school,
            'notes' => [],
            'coefficients' => [],
            'trimestre' => $trimestre,
            'moyennes_matiere' => [],
            'moyenne_generale' => null,
            'moyennes_sous_matieres' => [],
            'message' => "Aucune note enregistrée pour ce trimestre."
        ]);
    }

    // -----------------------
    // Traitement + conversion sur 20
    // -----------------------
    $notes = [];
    $coefficients = [];
    $levelMatterIds = [];

    foreach ($rows as $row) {

        // Détecter la base automatiquement
        $base = $detectBase($row->evaluation_type);

        // Convertir la note en nombre (sécurisé)
        $note = str_replace(',', '.', $row->note);
        $note = (float) $note;

        // Conversion finale sur 20
        $row->note_sur_20 = ($note * 20) / $base;

        // Regroupement
        $matiere = $row->matiere;
        $sub_matter_id = $row->sub_matter_id;

        if (!isset($notes[$matiere])) $notes[$matiere] = [];

        if ($sub_matter_id) {
            $notes[$matiere]['sous_matieres'][$sub_matter_id][] = $row;
        } else {
            $notes[$matiere]['principale'][] = $row;
        }

        $coefficients[$matiere] = $row->coefficient ?? 1;
        $levelMatterIds[$matiere] = $row->level_matter_id;
    }

    // -----------------------
    // Moyennes par matière
    // -----------------------
   // -----------------------
// Moyennes par matière (avec coefficients)
// -----------------------
// -----------------------
// Moyennes par matière (avec exclusion des NC)
// -----------------------
$moyennes_matiere = [];

$total_pondere = 0;
$total_coef = 0;

foreach ($levelMatterIds as $matiere => $level_matter_id) {

    $moyenne = DB::table('moyenne_matieres')
        ->where('inscription_id', $inscription->id)
        ->where('level_matter_id', $level_matter_id)
        ->where('cutting_school_year_id', $trimestre->id)
        ->value('moyenne');

    // Nettoyage
    if ($moyenne !== null) {
        $moyenne = trim($moyenne);
    }

    // Si la moyenne vaut 'NC' → on l'affiche mais on ne la calcule pas
    if ($moyenne !== null && strtolower($moyenne) !== 'nc') {

        // Conversion float
        $moyenne = (float) str_replace(',', '.', $moyenne);

        // Coefficient réel de la matière
        $coef = $coefficients[$matiere] ?? 1;

        // Stockage pour affichage
        $moyennes_matiere[$matiere] = round($moyenne, 2);

        // -----------------------------
        // CALCUL PONDÉRÉ
        // -----------------------------
        $total_pondere += ($moyenne * $coef);
        $total_coef += $coef;

    } else {
        // Pour l’affichage dans la liste des matières
        $moyennes_matiere[$matiere] = 'NC';
    }
}



    // -----------------------
    // Moyennes par sous-matière
    // -----------------------
    $moyennes_sous_matieres = [];

    foreach ($notes as $matiere => $groupes) {
        if (isset($groupes['sous_matieres'])) {
            foreach ($groupes['sous_matieres'] as $sub_matter_id => $listeSous) {
                $moyenne = DB::table('moyenne_sub_matters')
                    ->where('inscription_id', $inscription->id)
                    ->where('sub_matter_id', $sub_matter_id)
                    ->where('cutting_school_year_id', $trimestre->id)
                    ->value('moyenne');

                if ($moyenne !== null) {
                    $moyenne = str_replace(',', '.', $moyenne);
                    $moyenne = (float) $moyenne;
                }

                $moyennes_sous_matieres[$matiere][$sub_matter_id] = $moyenne ? round($moyenne, 2) : null;
            }
        }
    }

    // -----------------------
    // Moyenne générale
    
$moyenne_generale = $total_coef > 0 ? round($total_pondere / $total_coef, 2) : null;


    return view('notes.trimestre1', compact(
        'student',
        'school',
        'notes',
        'coefficients',
        'trimestre',
        'moyennes_matiere',
        'moyenne_generale',
        'moyennes_sous_matieres'
    ));
})->name('notes.trimestre1');

/*
|--------------------------------------------------------------------------
| 🧾 Route d’affichage des notes du 1er trimestre (via contrôleur)
|--------------------------------------------------------------------------
| Cette route utilise le contrôleur NoteController pour plus de clarté.
| Exemple de méthode :
|   NoteController::showTrimestre1($studentId)
|--------------------------------------------------------------------------
*/

Route::get('/notes/trimestre2', function () {

    $student = Auth::user();
    $school = $student->school()->with('setting')->first();
    $inscription = $student->inscriptions()->latest()->first();
    if (!$inscription) return "Aucune inscription trouvée pour cet élève.";

    $trimestre = DB::table('cuttings')->where('libelle', 'Trimestre 2')->first();
    if (!$trimestre) return "Trimestre 2 non défini.";

    $pdo = DB::connection()->getPdo();

    $detectBase = function ($type) {
        $type = strtolower($type);
        if (in_array($type, ['interro', 'interrogation'])) return 10;
        if (in_array($type, ['composition', 'examen'])) return 40;
        if (in_array($type, ['devoir', 'controle', 'contrôle'])) return 20;
        return 20;
    };

    $sql = "
    SELECT 
        m.id AS matter_id,
        m.libelle AS matiere,
        e.sub_matter_id,
        CASE e.sub_matter_id
            WHEN 1 THEN 'Composition Française'
            WHEN 2 THEN 'Orthographe-Grammaire'
            WHEN 3 THEN 'Expression Orale'
            ELSE NULL
        END AS sous_matiere,
        et.type AS evaluation_type,
        gn.note,
        e.notee,
        e.date_create,
        coef.valeur AS coefficient,
        lm.id AS level_matter_id
    FROM get_notes gn
    INNER JOIN evaluations e ON gn.evaluation_id = e.id
    INNER JOIN level_matter lm ON e.level_matter_id = lm.id
    INNER JOIN matters m ON lm.matter_id = m.id
    INNER JOIN coefficients coef ON lm.coefficient_id = coef.id
    INNER JOIN evaluation_types et ON e.evaluation_type_id = et.id
    INNER JOIN cutting_school_year csy ON e.cutting_school_year_id = csy.id
    WHERE gn.inscription_id = :inscription_id
      AND csy.cutting_id = :trimestre_id
    ORDER BY m.libelle ASC, e.sub_matter_id ASC, e.id ASC
    ";

    $stmt = $pdo->prepare($sql);
    $stmt->execute([
        'inscription_id' => $inscription->id,
        'trimestre_id' => $trimestre->id
    ]);

    $rows = $stmt->fetchAll(PDO::FETCH_OBJ);

    if (empty($rows)) {
        return view('notes.trimestre2', [
            'student' => $student,
            'school' => $school,
            'notes' => [],
            'coefficients' => [],
            'trimestre' => $trimestre,
            'moyennes_matiere' => [],
            'moyenne_generale' => null,
            'moyennes_sous_matieres' => [],
            'message' => "Aucune note enregistrée pour ce trimestre."
        ]);
    }

    $notes = [];
    $coefficients = [];
    $levelMatterIds = [];

    foreach ($rows as $row) {
        $base = $detectBase($row->evaluation_type);
        $note = (float) str_replace(',', '.', $row->note);
        $row->note_sur_20 = ($note * 20) / $base;

        $matiere = $row->matiere;
        $sub_matter_id = $row->sub_matter_id;

        if (!isset($notes[$matiere])) $notes[$matiere] = [];

        if ($sub_matter_id) {
            $notes[$matiere]['sous_matieres'][$sub_matter_id][] = $row;
        } else {
            $notes[$matiere]['principale'][] = $row;
        }

        $coefficients[$matiere] = $row->coefficient ?? 1;
        $levelMatterIds[$matiere] = $row->level_matter_id;
    }

    $moyennes_matiere = [];
    $total_pondere = 0;
    $total_coef = 0;

    foreach ($levelMatterIds as $matiere => $level_matter_id) {

        $moyenne = DB::table('moyenne_matieres')
            ->where('inscription_id', $inscription->id)
            ->where('level_matter_id', $level_matter_id)
            ->where('cutting_school_year_id', $trimestre->id)
            ->value('moyenne');

        if ($moyenne !== null && strtolower(trim($moyenne)) !== 'nc') {

            $moyenne = (float) str_replace(',', '.', $moyenne);
            $coef = $coefficients[$matiere] ?? 1;

            $moyennes_matiere[$matiere] = round($moyenne, 2);
            $total_pondere += ($moyenne * $coef);
            $total_coef += $coef;

        } else {
            $moyennes_matiere[$matiere] = 'NC';
        }
    }

    $moyennes_sous_matieres = [];

    foreach ($notes as $matiere => $groupes) {
        if (isset($groupes['sous_matieres'])) {
            foreach ($groupes['sous_matieres'] as $sub_matter_id => $listeSous) {

                $moyenne = DB::table('moyenne_sub_matters')
                    ->where('inscription_id', $inscription->id)
                    ->where('sub_matter_id', $sub_matter_id)
                    ->where('cutting_school_year_id', $trimestre->id)
                    ->value('moyenne');

                if ($moyenne !== null) {
                    $moyenne = (float) str_replace(',', '.', $moyenne);
                }

                $moyennes_sous_matieres[$matiere][$sub_matter_id] = 
                    $moyenne ? round($moyenne, 2) : null;
            }
        }
    }

    $moyenne_generale = $total_coef > 0 
        ? round($total_pondere / $total_coef, 2)
        : null;

    return view('notes.trimestre2', compact(
        'student',
        'school',
        'notes',
        'coefficients',
        'trimestre',
        'moyennes_matiere',
        'moyenne_generale',
        'moyennes_sous_matieres'
    ));

})->name('notes.trimestre2');





// ---- AUTRES ROUTES ----
// Route::get('/', ...);
// Route::middleware([...])->group(function() { ... });

// ===============================
// 🚀 ROUTE POUR REFAIRE LES MOYENNES PREMIER TRIMESTRE
// ===============================
// Route Laravel
Route::get('/admin/recalcul-moyennes', function () {


$trimestre = DB::table('cuttings')->where('libelle', 'Trimestre 1')->first();
if (!$trimestre) return "Trimestre 1 non défini.";

// Récupérer tous les élèves avec leur classe_id
$inscriptions = DB::table('inscriptions')
    ->select('id', 'classe_id')
    ->get();

$pdo = DB::connection()->getPdo();

$detectBase = function ($type) {
    $type = strtolower($type);
    if (in_array($type, ['interro', 'interrogation'])) return 10;
    if (in_array($type, ['composition', 'examen'])) return 40;
    if (in_array($type, ['devoir', 'controle', 'contrôle'])) return 20;
    return 20;
};

$resultats = [];
$moyennes_par_classe = [];

foreach ($inscriptions as $inscription) {

    $sql = "
    SELECT 
        m.id AS matter_id,
        m.libelle AS matiere,
        e.sub_matter_id,
        et.type AS evaluation_type,
        gn.note,
        coef.valeur AS coefficient,
        lm.id AS level_matter_id
    FROM get_notes gn
    INNER JOIN evaluations e ON gn.evaluation_id = e.id
    INNER JOIN level_matter lm ON e.level_matter_id = lm.id
    INNER JOIN matters m ON lm.matter_id = m.id
    INNER JOIN coefficients coef ON lm.coefficient_id = coef.id
    INNER JOIN evaluation_types et ON e.evaluation_type_id = et.id
    INNER JOIN cutting_school_year csy ON e.cutting_school_year_id = csy.id
    WHERE gn.inscription_id = :inscription_id
      AND csy.cutting_id = :trimestre_id
    ";

    $stmt = $pdo->prepare($sql);
    $stmt->execute([
        'inscription_id' => $inscription->id,
        'trimestre_id' => $trimestre->id
    ]);

    $rows = $stmt->fetchAll(PDO::FETCH_OBJ);

    if (empty($rows)) {
        $resultats[] = [
            'inscription_id' => $inscription->id,
            'classe_id' => $inscription->classe_id,
            'moyenne_generale' => null,
            'matieres' => [],
            'rang' => null
        ];
        continue;
    }

    $matieres_calculees = [];
    $total_pondere = 0;
    $total_coef = 0;
    $notes_par_matiere = [];
    $coef_par_matiere = [];

    foreach ($rows as $row) {
        $note_str = trim($row->note);
        if (strtolower($note_str) === 'nc' || $note_str === null || $note_str === '') {
            continue;
        }

        $base = $detectBase($row->evaluation_type);
        $note = (float) str_replace(',', '.', $note_str);
        $note_sur_20 = ($note * 20) / $base;

        $matiere = $row->matiere;
        $notes_par_matiere[$matiere][] = $note_sur_20;
        $coef_par_matiere[$matiere] = $row->coefficient ?? 1;
    }

    foreach ($notes_par_matiere as $matiere => $notes_mat) {
        $moyenne_matiere = array_sum($notes_mat) / count($notes_mat);
        $coef = $coef_par_matiere[$matiere];

        $matieres_calculees[] = [
            'matiere' => $matiere,
            'moyenne' => round($moyenne_matiere, 2),
            'coef' => $coef
        ];

        $total_pondere += ($moyenne_matiere * $coef);
        $total_coef += $coef;
    }

    $moyenne_generale = $total_coef > 0 ? round($total_pondere / $total_coef, 2) : null;

    $moyennes_par_classe[$inscription->classe_id][] = [
        'inscription_id' => $inscription->id,
        'moyenne_generale' => $moyenne_generale
    ];

    $resultats[] = [
        'inscription_id' => $inscription->id,
        'classe_id' => $inscription->classe_id,
        'matieres' => $matieres_calculees,
        'moyenne_generale' => $moyenne_generale,
        'rang' => null
    ];
}

// Calcul du rang avec ex-aequo
foreach ($moyennes_par_classe as $classe_id => $liste_eleves) {
    usort($liste_eleves, fn($a, $b) => $b['moyenne_generale'] <=> $a['moyenne_generale']);

    $rang = 1;
    $compteur = 1;
    $precedent = null;

    foreach ($liste_eleves as $eleve) {
        foreach ($resultats as &$res) {
            if ($res['inscription_id'] == $eleve['inscription_id']) {
                if ($precedent !== null && $eleve['moyenne_generale'] == $precedent) {
                    $res['rang'] = $rang; // ex-aequo
                } else {
                    $rang = $compteur; // nouveau rang
                    $res['rang'] = $rang;
                }
                $precedent = $eleve['moyenne_generale'];
                $compteur++;
                break;
            }
        }
    }
}

return view('admin.recalcul-moyennes', compact('resultats', 'trimestre'));

});

// ===============================
// 🚀 ROUTE POUR REFAIRE LES MOYENNES PREMIER TRIMESTRE
// ===============================
// Route Laravel

Route::get('/admin/recalcul-moyennes-trimestre2', function () {

$trimestre = DB::table('cuttings')->where('libelle', 'Trimestre 2')->first();
if (!$trimestre) return "Trimestre 2 non défini.";

$inscriptions = DB::table('inscriptions')
    ->select('id', 'classe_id')
    ->get();

$pdo = DB::connection()->getPdo();

$detectBase = function ($type) {
    $type = strtolower($type);
    if (in_array($type, ['interro', 'interrogation'])) return 10;
    if (in_array($type, ['composition', 'examen'])) return 40;
    if (in_array($type, ['devoir', 'controle', 'contrôle'])) return 20;
    return 20;
};

$resultats = [];
$moyennes_par_classe = [];

foreach ($inscriptions as $inscription) {

    $sql = "
    SELECT 
        m.id AS matter_id,
        m.libelle AS matiere,
        e.sub_matter_id,
        et.type AS evaluation_type,
        gn.note,
        coef.valeur AS coefficient,
        lm.id AS level_matter_id
    FROM get_notes gn
    INNER JOIN evaluations e ON gn.evaluation_id = e.id
    INNER JOIN level_matter lm ON e.level_matter_id = lm.id
    INNER JOIN matters m ON lm.matter_id = m.id
    INNER JOIN coefficients coef ON lm.coefficient_id = coef.id
    INNER JOIN evaluation_types et ON e.evaluation_type_id = et.id
    INNER JOIN cutting_school_year csy ON e.cutting_school_year_id = csy.id
    WHERE gn.inscription_id = :inscription_id
      AND csy.cutting_id = :trimestre_id
    ";

    $stmt = $pdo->prepare($sql);
    $stmt->execute([
        'inscription_id' => $inscription->id,
        'trimestre_id' => $trimestre->id
    ]);

    $rows = $stmt->fetchAll(PDO::FETCH_OBJ);

    if (empty($rows)) {
        $resultats[] = [
            'inscription_id' => $inscription->id,
            'classe_id' => $inscription->classe_id,
            'moyenne_generale' => null,
            'matieres' => [],
            'rang' => null
        ];
        continue;
    }

    $total_pondere = 0;
    $total_coef = 0;
    $notes_par_matiere = [];
    $coef_par_matiere = [];

    foreach ($rows as $row) {

        $note_str = trim($row->note);
        if (strtolower($note_str) === 'nc' || $note_str === '') continue;

        $base = $detectBase($row->evaluation_type);
        $note = (float) str_replace(',', '.', $note_str);
        $note_sur_20 = ($note * 20) / $base;

        $matiere = $row->matiere;
        $notes_par_matiere[$matiere][] = $note_sur_20;
        $coef_par_matiere[$matiere] = $row->coefficient ?? 1;
    }

    foreach ($notes_par_matiere as $matiere => $notes_mat) {

        $moyenne_matiere = array_sum($notes_mat) / count($notes_mat);
        $coef = $coef_par_matiere[$matiere];

        $total_pondere += ($moyenne_matiere * $coef);
        $total_coef += $coef;
    }

    $moyenne_generale = $total_coef > 0
        ? round($total_pondere / $total_coef, 2)
        : null;

    $moyennes_par_classe[$inscription->classe_id][] = [
        'inscription_id' => $inscription->id,
        'moyenne_generale' => $moyenne_generale
    ];

    $resultats[] = [
        'inscription_id' => $inscription->id,
        'classe_id' => $inscription->classe_id,
        'moyenne_generale' => $moyenne_generale,
        'rang' => null
    ];
}

foreach ($moyennes_par_classe as $classe_id => $liste_eleves) {

    usort($liste_eleves, fn($a, $b) => 
        $b['moyenne_generale'] <=> $a['moyenne_generale']
    );

    $rang = 1;
    $compteur = 1;
    $precedent = null;

    foreach ($liste_eleves as $eleve) {
        foreach ($resultats as &$res) {
            if ($res['inscription_id'] == $eleve['inscription_id']) {

                if ($precedent !== null && 
                    $eleve['moyenne_generale'] == $precedent) {
                    $res['rang'] = $rang;
                } else {
                    $rang = $compteur;
                    $res['rang'] = $rang;
                }

                $precedent = $eleve['moyenne_generale'];
                $compteur++;
                break;
            }
        }
    }
}

return view('admin.recalcul-moyennes', compact('resultats', 'trimestre'));

});








/*
|--------------------------------------------------------------------------
| 📅 Route de l’emploi du temps
|--------------------------------------------------------------------------
| - Vérifie l’authentification
| - Récupère les cours du matin et de l’après-midi
| - Trie par jour de la semaine et par ordre horaire
| - Regroupe les cours pour un affichage clair dans la vue "emploi_du_temps"
|--------------------------------------------------------------------------
*/
Route::get('/emploi-du-temps', function () {
    ensureAuthenticated();

    $student = Auth::user();
    $school  = $student->school()->with('setting')->first();
    $classe  = $student->currentClass;
    $classeId = $classe?->id;

    // Si pas de classe, retourne des collections vides
    if (!$classeId) {
        $mornings = collect();
        $afters = collect();
    } else {
        // Cours du matin
        $mornings = TimeMorning::with(['matter', 'morningTime', 'weekDay'])
            ->where('school_id', $school->id)
            ->where('classe_id', $classeId)
            ->orderBy('week_day_id')
            ->orderBy('morning_time_id')
            ->get()
            ->groupBy(fn($session) => $session->week_day_id);

        // Cours de l'après-midi
        $afters = TimeAfter::with(['matter', 'afterNoonTime', 'weekDay'])
            ->where('school_id', $school->id)
            ->where('classe_id', $classeId)
            ->orderBy('week_day_id')
            ->orderBy('after_noon_time_id')
            ->get()
            ->groupBy(fn($session) => $session->week_day_id);
    }

    return view('emploi_du_temps', compact('student', 'school', 'mornings', 'afters'));
})->name('emploi.du.temps');

/*
|--------------------------------------------------------------------------
| 🚪 Déconnexion
|--------------------------------------------------------------------------
| Supprime la session et redirige vers la page de connexion principale.
|--------------------------------------------------------------------------
*/
Route::get('/logout', function () {
    Auth::logout();
    session()->flush();
    return redirect('https://educas-nsa.net/academie/connexion');
})->name('logout');

/*
|--------------------------------------------------------------------------
| 🌐 Routes publiques
|--------------------------------------------------------------------------
| Ces routes ne nécessitent pas d’authentification.
|--------------------------------------------------------------------------
*/
Route::get('/about', function () {
    return view('about');
})->name('about');

/*
|--------------------------------------------------------------------------
| 🔑 Route de connexion par matricule
|--------------------------------------------------------------------------
| Cette route :
| - Vérifie le matricule transmis en GET
| - Connecte automatiquement l’élève correspondant
| - Redirige vers l’accueil s’il existe
| - Sinon, redirige vers la page de connexion EDUCAS-NSA
|--------------------------------------------------------------------------
*/
Route::get('/login-redirect', function (Request $request) {
    $student = Student::where('matricule', $request->matricule)
        ->with('school.setting')
        ->first();

    if ($student) {
        Auth::login($student);
        session(['school_id' => $student->school_id]);
        return redirect()->route('welcome');
    }

    return redirect('https://educas-nsa.net/academie/connexion')
        ->withErrors(['matricule' => 'Matricule invalide']);
})->name('login.redirect');
